<?php
// PHP 8.2 Compatibility Test
echo "<h2>PHP 8.2 Compatibility Test</h2>";
echo "<style>body{font-family:Arial;margin:20px;} .ok{color:green;} .error{color:red;} .warning{color:orange;} .info{color:blue;}</style>";

echo "<h3>PHP Version Information:</h3>";
echo "<div class='ok'>PHP Version: " . PHP_VERSION . "</div>";
echo "<div class='info'>PHP SAPI: " . php_sapi_name() . "</div>";
echo "<div class='info'>Server: " . $_SERVER['SERVER_SOFTWARE'] . "</div>";

echo "<h3>Required Extensions:</h3>";
$required_extensions = [
    'pdo' => 'PDO (PHP Data Objects)',
    'pdo_mysql' => 'PDO MySQL Driver',
    'session' => 'Session Support',
    'json' => 'JSON Support',
    'mbstring' => 'Multibyte String Support',
    'openssl' => 'OpenSSL Support'
];

foreach ($required_extensions as $ext => $desc) {
    if (extension_loaded($ext)) {
        echo "<div class='ok'>✓ $desc ($ext)</div>";
    } else {
        echo "<div class='error'>✗ $desc ($ext) - NOT LOADED</div>";
    }
}

echo "<h3>PHP 8.2 Features Test:</h3>";

// Test password hashing (PHP 8.2 compatible)
try {
    $password = 'test123';
    $hash = password_hash($password, PASSWORD_DEFAULT);
    if (password_verify($password, $hash)) {
        echo "<div class='ok'>✓ Password hashing works correctly</div>";
    } else {
        echo "<div class='error'>✗ Password hashing failed</div>";
    }
} catch (Exception $e) {
    echo "<div class='error'>✗ Password hashing error: " . $e->getMessage() . "</div>";
}

// Test PDO with MySQL
try {
    $pdo = new PDO("mysql:host=localhost", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "<div class='ok'>✓ PDO MySQL connection successful</div>";
    
    // Test prepared statements
    $stmt = $pdo->prepare("SELECT 1 as test");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($result['test'] == 1) {
        echo "<div class='ok'>✓ Prepared statements work correctly</div>";
    }
} catch (PDOException $e) {
    echo "<div class='error'>✗ PDO MySQL error: " . $e->getMessage() . "</div>";
}

// Test sessions
session_start();
if (session_status() === PHP_SESSION_ACTIVE) {
    echo "<div class='ok'>✓ Sessions work correctly</div>";
} else {
    echo "<div class='error'>✗ Sessions not working</div>";
}

// Test file operations
$test_file = 'php82_test_write.txt';
if (file_put_contents($test_file, 'PHP 8.2 test')) {
    echo "<div class='ok'>✓ File write operations work</div>";
    unlink($test_file);
} else {
    echo "<div class='error'>✗ File write operations failed</div>";
}

// Test JSON
$test_data = ['name' => 'test', 'version' => '8.2'];
$json = json_encode($test_data);
$decoded = json_decode($json, true);
if ($decoded['name'] === 'test') {
    echo "<div class='ok'>✓ JSON operations work correctly</div>";
} else {
    echo "<div class='error'>✗ JSON operations failed</div>";
}

// Test error handling
try {
    throw new Exception('Test exception');
} catch (Exception $e) {
    echo "<div class='ok'>✓ Exception handling works correctly</div>";
}

echo "<h3>PHP 8.2 Specific Features:</h3>";

// Test readonly classes (PHP 8.2 feature)
if (version_compare(PHP_VERSION, '8.2.0', '>=')) {
    echo "<div class='ok'>✓ PHP 8.2+ features available</div>";
    
    // Test null coalescing assignment
    $test_var = null;
    $test_var ??= 'default_value';
    if ($test_var === 'default_value') {
        echo "<div class='ok'>✓ Null coalescing assignment works</div>";
    }
    
    // Test match expression (PHP 8.0+)
    $test_value = 1;
    $result = match($test_value) {
        1 => 'one',
        2 => 'two',
        default => 'other'
    };
    if ($result === 'one') {
        echo "<div class='ok'>✓ Match expression works</div>";
    }
} else {
    echo "<div class='warning'>⚠ PHP version is below 8.2, some features may not be available</div>";
}

echo "<h3>System Requirements Check:</h3>";
echo "<div class='info'>Memory Limit: " . ini_get('memory_limit') . "</div>";
echo "<div class='info'>Max Execution Time: " . ini_get('max_execution_time') . " seconds</div>";
echo "<div class='info'>Upload Max Filesize: " . ini_get('upload_max_filesize') . "</div>";
echo "<div class='info'>Post Max Size: " . ini_get('post_max_size') . "</div>";

echo "<h3>Security Features:</h3>";
if (ini_get('display_errors') == 0) {
    echo "<div class='ok'>✓ Display errors is disabled (good for production)</div>";
} else {
    echo "<div class='warning'>⚠ Display errors is enabled (good for development)</div>";
}

if (ini_get('expose_php') == 0) {
    echo "<div class='ok'>✓ PHP version is not exposed</div>";
} else {
    echo "<div class='warning'>⚠ PHP version is exposed in headers</div>";
}

echo "<hr>";
echo "<h3>Conclusion:</h3>";
if (version_compare(PHP_VERSION, '8.2.0', '>=')) {
    echo "<div class='ok'><strong>✓ Your PHP 8.2 setup is perfect for the Student Management System!</strong></div>";
    echo "<p>The system will run smoothly with better performance and security features.</p>";
} else {
    echo "<div class='warning'><strong>⚠ Your PHP version is below 8.2</strong></div>";
    echo "<p>The system will still work, but you might miss some performance improvements.</p>";
}

echo "<p><a href='setup_admin.php'>Setup Admin User</a> | <a href='login.php'>Go to Login</a> | <a href='install.php'>Run Installation</a></p>";
?>
